%% Read data file (.tdms) for CAE acoustic camera
% Input needed: data.dir = full-directory-to-file-name (f.e. 'U:\Schiphol taxibot noise\2 - Data MC taxibot\2 - Data\MC1_Corendon_L1\measurements\MC1_Corendon_L1_R1_F1_12-00_taxibot.tdms')
% data.start_time and data.n_samples
% Outputs structs: data, info, video, config
% Bieke von den Hoff
% PhD student ANCE - Aerospace Engineering - TU Delft 
% Version January 18, 2021

% data.dir = 'U:\Measurement Campaign Schiphol 2019\2 - Information on acoustic cameras\CAE systems information\Matlab code\Meetingroom_outline-cam\measurements\Source_middle.tdms';
% data.start_time = 9;
% data.n_samples = 8192; % or any other power of 2 that is <=8192

%% Call folders
% Unless TDMS-reader is placed in same folder, need:
addpath('U:\Measurement Campaign Schiphol 2019\2 - Information on acoustic cameras\CAE systems information\Matlab code\post_process_CAE_array');

% File name
file_path = fullfile(data.dir);

%% Read the TDMS file
% finalOutput contains direct information of data
% metaStruct contains all information, but in deeper layers
finalOutput = TDMS_readTDMSFile(file_path); % [finalOutput,metaStruct] 

%% Create info struct
% Only useful outputs are stored in info struct
info.filename = finalOutput.propValues{1,1}{1,1};
info.fs = finalOutput.propValues{1,1}{1,2}; % Hz, this is 1.6340 lower than the fs = 48e3 stated by CAE (can be confirmed with the waveform dt (finalOutput.propValues{1,1}{1,12})
info.pref = finalOutput.propValues{1,3}{1,17}; % Pa, taken from data of Mic000
info.calibrationdate = finalOutput.propValues{1,3}{1,18}; % taken from data of Mic000
info.T = finalOutput.propValues{1,1}{1,3}; % degrees C, set by researcher during recording
info.d = finalOutput.propValues{1,1}{1,4}; % m, as set by researcher during recording
info.comment = finalOutput.propValues{1,1}{1,5}; % set by researcher during recording
info.author = finalOutput.propValues{1,1}{1,6}; % set by researcher during recording
info.datetime = finalOutput.propValues{1,1}{1,12}; % dd-mm-yyyy-hh:mm
info.windx = finalOutput.propValues{1,1}{1,13}; % m/s, set by researcher during recording
info.windy = finalOutput.propValues{1,1}{1,15}; % m/s, set by researcher during recording

disp('WARNING - temperature, wind, and distance are as input during measurement, might need to be updated!')

%% Create config struct
% Read microphone positions
for i = 1:112
    config.channel(i) = string(finalOutput.propValues{1,i+2}{1,14}); % Mic###, note Mic000 is not the most centered microphone!
    config.x(i) = finalOutput.propValues{1,i+2}{1,7};
    config.y(i) = finalOutput.propValues{1,i+2}{1,8};
    config.z(i) = finalOutput.propValues{1,i+2}{1,9};
    config.datarow(i) = i; % to see which mic matches with which row, in principle Mic000 = 1, Mic001 = 2 etc.
end 

%% Create data struct
data.data_length = length(finalOutput.data{3});

% Read data gathered
data.full_data = zeros(112,data.data_length); % initialize the data vector
for i = 1:112 % i = channel number
    data.full_data(i,:) = finalOutput.data{i+2}; % contains recorded pressure data 
end

data.full_time = 0:1/info.fs:(data.data_length-1)/info.fs; % time stamp per pressure data point

% Create a selection of data 
data.start_sample = floor(data.start_time*info.fs);
if data.start_sample == 0
    data.start_sample = 1;
end
% data.end_sample = ceil(data.end_time*info.fs);
data.end_sample = data.n_samples-1+data.start_sample;
data.end_time = data.end_sample/info.fs;

data.selected_data = data.full_data(:,data.start_sample:data.end_sample); % contains pressure data for the selected time interval
data.selected_time = data.full_time(data.start_sample:data.end_sample); % contains time stamps for the selected time interval

%% Frequency calibration correction on data
load('U:\Measurement Campaign Schiphol 2019\2 - Information on acoustic cameras\CAE systems information\Matlab code\Frequency_correction.mat')

I_normal = 1:length(config.x);
I_middle = [16 32 48 64 80 96 112]; % middle mics that have a different correction curve
I_normal(I_middle) = []; % I_normal is now all mics except the middle mics 

zeropad = input('Zeropadding for correction: (1) = none, (2) = once zeropadded, etc: ');
block_overlap = []; % 50% overlap
block_size = data.n_samples;

for i = 1:length(config.x)
    [X_PSD,f] = pwelch(data.selected_data(i,:),hann(block_size),block_overlap,zeropad*block_size,info.fs); % X_PSD = (abs(Xdft)).^2;
    PSD(:,i) = X_PSD;
end

df = f(2)-f(1);
SPL = 10*log10(df*PSD/(info.pref^2));

% Calibrate the PSD and SPL
match_FD = df/(freq_cal(2)-freq_cal(1));
if match_FD < 1 % This is true when (zeropad*block_size) > 4096
    disp('Warning: Calibration is interpolated!')
    cal_mid_factorPSD = interp1(freq_cal,cal_mid_factorPSD,f);
    cal_out_factorPSD = interp1(freq_cal,cal_out_factorPSD,f);
    match_FD = 1; % to make the matching work in line 105 and 106. For match_FD<1 it repeats values
end

% Divide by the factor, because the correction = CAE/REFERENCE, so to get back to REFERENCE, need to divide by correction
PSD_cal(:,I_middle) = PSD(:,I_middle)./cal_mid_factorPSD(1:match_FD:end);
PSD_cal(:,I_normal) = PSD(:,I_normal)./cal_out_factorPSD(1:match_FD:end);

SPL_cal = 10*log10(df*PSD_cal/(info.pref^2)); % Also equals SPL-cal_dB

%% Time domain calibration - two sided, no weighting, no zeropadding!
nfft_cal = length(data.selected_data);
Xdft = fft(data.selected_data,nfft_cal,2);
df_fft = info.fs/nfft_cal;
freq_TD_cal = 0:df_fft:(nfft_cal-1)*df_fft;

% Get positions in freq (same as cor_dB or cor_factor) where f=freq, to apply correction at different settings of fft 
match_TD = df_fft/(freq_two_cal(2)-freq_two_cal(1)); % Equals 1 (8192 samples), 2 (4096 samples), 4 (2048 samples)

Xdft_calibrated(I_middle,:) = Xdft(I_middle,:)./cal_mid_factorFT_two(1:match_TD:end)';
Xdft_calibrated(I_normal,:) = Xdft(I_normal,:)./cal_out_factorFT_two(1:match_TD:end)';
for i = 1:112
    data.selected_data_cal(i,:) = real(ifft(Xdft_calibrated(i,:))); % contains calibrated pressure data for selected time interval
end

%% Create video struct
video.size_x = finalOutput.propValues{1,126}{1,9}; % taken from data of first video frame
video.size_y = finalOutput.propValues{1,126}{1,10}; % taken from data of first video frame
video.n_frames = 0;
video.frames = [];
for i = 1:length(finalOutput.data)
    if length(finalOutput.propValues{1,i})>=19 && (string(finalOutput.propValues{1,i}{1,17})=='record' || string(finalOutput.propValues{1,i}{1,17})=='first to record')
        video.n_frames = video.n_frames + 1;
        video.frames = [video.frames i];
    end
end

for i = 1:video.n_frames % the last 4 images have 'no purpose' as recording type. data of 126 has 'first to record' as recording type and is also ommitted.
    if size(finalOutput.data{1,video.frames(i)},2) > 90240 % sometimes double saved
        finalOutput.data{1,video.frames(i)} = finalOutput.data{1,video.frames(i)}(1:90240);
    end
    video.images(:,:,i) = imrotate(fliplr(reshape(finalOutput.data{1,video.frames(i)},376,240)),90);
    video.timestamp(i) = double(finalOutput.propValues{1,video.frames(i)}{1,6})/1000;
end

video.selected_frames = find(video.timestamp>data.start_time & video.timestamp<data.end_time);
video.selected_video = video.images(:,:,video.selected_frames);
video.selected_time = video.timestamp(video.selected_frames);

video.frame_rate = video.n_frames/(video.timestamp(end)); % Roughly 55.1526 frames/sec
video.offset = data.full_time(end) - video.timestamp(end) ; % How much earlier does the video stop w.r.t. mic measurement?
% Often the video is 6 frames short, so it is more accurate to start selecting frames from beginning

% % If you want to check out the video of the measurements, run the following snippet of code - adapted from DEMO optical camera
% fig = figure(1);
% tb = uicontrol(fig, 'Style', 'togglebutton', 'String', 'Start/stop');
% drawnow;
% for i=1:size(video.images,3) 
%     h1 = imshow(video.images(:,:,i));
%     text(-10,-10,num2str(i)) % plot frame number in top left corner
%     while true
%         drawnow;
%          if (get(tb, 'Value')==1); break; end
%     end
% end

%% End read data file
disp('Data, info, config, and video struct created')
clearvars -except data info config video % Removed because relevant data is saved in new structs


